<?php

namespace App\Livewire;

use Livewire\Component;
use Illuminate\Support\Facades\Auth;
use App\Models\Transaction;
use Livewire\WithPagination;
use App\Models\Address;
use SimpleSoftwareIO\QrCode\Facades\QrCode;

class Wallet extends Component
{
    use WithPagination;

    public $showDepositModal = false;
    public $showWithdrawModal = false;
    public $showDepositAddressModal = false;
    public $amount;
    public $selectedCurrency = 'USD';
    public $canWithdraw = false;
    public $toAddress;

    protected $rules = [
        'amount' => 'required|numeric|min:10',
    ];

    public function openDepositModal()
    {
        $this->showDepositModal = true;
    }

    public function openWithdrawModal()
    {
        $this->showWithdrawModal = true;
    }

    public function closeModal()
    {
        $this->showDepositModal = false;
        $this->showWithdrawModal = false;
        $this->showDepositAddressModal = false;
        $this->reset(['amount']);
    }

    public function initiateDeposit()
    {
        $this->validate();

        // Create a pending deposit transaction
        $transaction = Transaction::create([
            'user_id' => Auth::id(),
            'type' => 'deposit',
            'amount' => $this->amount,
            'confirmed' => 0
        ]);

        $this->showDepositModal = false;
        $this->showDepositAddressModal = true;
    }

    public function withdraw()
    {
        $this->validate();

        // Check if user has sufficient balance
        $balance = Auth::user()->confirmedBalance();

        if ($balance < $this->amount) {
            session()->flash('error', 'Insufficient balance.');
            return;
        }

        // Create a pending withdrawal transaction
        Transaction::create([
            'user_id' => Auth::id(),
            'type' => 'withdraw',
            'amount' => $this->amount,
            'to_address' => $this->toAddress,
            'confirmed' => 0
        ]);

        session()->flash('message', 'Withdrawal request submitted successfully.');
        $this->closeModal();
    }

    public function updatedAmount()
    {
        if (!$this->amount) {
            $this->canWithdraw = false;
            return;
        }
        
        $this->canWithdraw = $this->amount <= Auth::user()->confirmedBalance();
    }

    public function rules()
    {
        $rules = [
            'amount' => 'required|numeric|min:10',
        ];

        // Only require toAddress for withdrawals
        if ($this->showWithdrawModal) {
            $rules['toAddress'] = 'required|string|min:26|max:45';
            $rules['amount'] = 'required|numeric|min:10|max:' . Auth::user()->confirmedBalance();
        }

        return $rules;
    }

    public function render()
    {
        $user = Auth::user();
        $transactions = Transaction::where('user_id', $user->id)->latest()->get();

        return view('livewire.wallet', [
            'user' => $user,
            'transactions' => $transactions,
        ]);
    }
} 