<?php

namespace App\Livewire;

use Livewire\Component;
use Livewire\WithPagination;
use App\Models\PromoCode;

class AdminPromoCodes extends Component
{
    use WithPagination;

    
    public $sortField = 'id';
    public $sortDirection = 'desc';
    public $showAddModal = false;
    public $showEditModal = false;
    public $showDeleteModal = false;
    public $deletingPromoCodeId = null;
    public $code = '';
    public $description = '';
    public $editingPromoCodeId = null;
    public $perPage = 10;
    public $search = '';
    
    protected $queryString = ['sortField', 'sortDirection', 'search'];
    
    protected $rules = [
        'code' => 'required|string|min:3|max:50',
        'description' => 'nullable|string|max:255',
    ];

    // Tell Livewire to use Bootstrap pagination theme
    protected $paginationTheme = 'bootstrap';

    // Reset pagination when search changes
    public function updatingSearch()
    {
        $this->resetPage();
    }

    // Reset pagination when per page changes
    public function updatingPerPage()
    {
        $this->resetPage();
    }

    public function sortBy($field)
    {
        if ($this->sortField === $field) {
            $this->sortDirection = $this->sortDirection === 'asc' ? 'desc' : 'asc';
        } else {
            $this->sortDirection = 'asc';
        }
        $this->sortField = $field;
    }

    public function openAddModal()
    {
        $this->reset(['code', 'description']);
        $this->showAddModal = true;
    }

    public function createPromoCode()
    {
        $this->validate();

        PromoCode::create([
            'code' => $this->code,
            'description' => $this->description,
        ]);

        $this->reset(['code', 'description']);
        $this->showAddModal = false;
        session()->flash('message', 'Promo code created successfully.');
    }

    public function openEditModal($id)
    {
        $promoCode = PromoCode::findOrFail($id);
        $this->editingPromoCodeId = $id;
        $this->code = $promoCode->code;
        $this->description = $promoCode->description;
        $this->showEditModal = true;
    }

    public function updatePromoCode()
    {
        $this->validate();

        $promoCode = PromoCode::findOrFail($this->editingPromoCodeId);
        $promoCode->update([
            'code' => $this->code,
            'description' => $this->description,
        ]);

        $this->reset(['code', 'description', 'editingPromoCodeId']);
        $this->showEditModal = false;
        session()->flash('message', 'Promo code updated successfully.');
    }

    public function confirmDelete($id)
    {
        $this->deletingPromoCodeId = $id;
        $this->showDeleteModal = true;
    }

    public function deletePromoCode()
    {
        if ($this->deletingPromoCodeId) {
            PromoCode::find($this->deletingPromoCodeId)->delete();
            $this->showDeleteModal = false;
            $this->deletingPromoCodeId = null;
            session()->flash('message', 'Promo code deleted successfully.');
        }
    }

    public function render()
    {
        $query = PromoCode::query();
        
        if ($this->search) {
            $query->where('code', 'like', '%' . $this->search . '%')
                  ->orWhere('description', 'like', '%' . $this->search . '%');
        }
        
        $query->orderBy($this->sortField, $this->sortDirection);

        return view('livewire.admin-promo-codes', [
            'promoCodes' => $query->paginate($this->perPage),
        ]);
    }
} 